<?php
// verify_and_download.php
session_start();
require 'db_connection.php'; // Your PDO connection file

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    die("Invalid request method.");
}

$access_token = htmlspecialchars(trim($_POST['access_token']));
$entered_otp = htmlspecialchars(trim($_POST['otp_code']));

// 1. Validate the Share Token & Get Document Details
$stmt = $pdo->prepare("
    SELECT ds.share_id, ds.document_id, ds.recipient_email, ds.recipient_name, d.title, d.original_file_path 
    FROM document_shares ds
    JOIN documents d ON ds.document_id = d.document_id
    WHERE ds.access_token = ? AND ds.status = 'shared'
");
$stmt->execute([$access_token]);
$share_data = $stmt->fetch();

if (!$share_data) {
    die("Error: Invalid link or document already processed.");
}

$email = $share_data['recipient_email'];

// 2. Verify the OTP
$otp_stmt = $pdo->prepare("
    SELECT token_id FROM otp_tokens 
    WHERE email = ? AND otp_code = ? AND is_used = 0 AND expires_at > NOW() 
    ORDER BY created_at DESC LIMIT 1
");
$otp_stmt->execute([$email, $entered_otp]);
$valid_otp = $otp_stmt->fetch();

if (!$valid_otp) {
    // Log the failed attempt
    $ip = $_SERVER['REMOTE_ADDR'];
    $pdo->prepare("INSERT INTO audit_logs (guest_email, action, ip_address) VALUES (?, 'Failed_Download_OTP', ?)")->execute([$email, $ip]);
    die("Error: Invalid or expired OTP code. Please request a new link.");
}

// 3. THE BURN: Mark OTP as used and update Share Status
$pdo->prepare("UPDATE otp_tokens SET is_used = 1 WHERE token_id = ?")->execute([$valid_otp['token_id']]);
$pdo->prepare("UPDATE document_shares SET status = 'downloaded', updated_at = NOW() WHERE share_id = ?")->execute([$share_data['share_id']]);

// Log the successful download
$ip = $_SERVER['REMOTE_ADDR'];
$pdo->prepare("INSERT INTO audit_logs (guest_email, document_id, action, ip_address) VALUES (?, ?, 'Document_Downloaded', ?)")
    ->execute([$email, $share_data['document_id'], $ip]);

// 4. File Preparation (The Steganography Phase)
$original_file = $share_data['original_file_path']; // e.g., '/var/www/secure_uploads/doc_45.docx'
if (!file_exists($original_file)) {
    die("Error: Source document missing on server.");
}

// Create a temporary copy to modify so we don't alter the master NCDC file
$temp_dir = sys_get_temp_dir();
$temp_file = $temp_dir . '/' . uniqid('ncdc_') . '_' . basename($original_file);
copy($original_file, $temp_file);

// 5. Inject the Hidden Tracker using ZipArchive
$zip = new ZipArchive;
if ($zip->open($temp_file) === TRUE) {
    
    // Create the digital fingerprint
    $tracker_data = [
        'share_id' => $share_data['share_id'],
        'recipient' => $share_data['recipient_name'],
        'email' => $email,
        'downloaded_at' => date('Y-m-d H:i:s'),
        'ip_address' => $ip,
        'ncdc_auth_token' => hash('sha256', $share_data['share_id'] . $email . 'SECRET_SALT') 
    ];
    
    // Convert array to a clean XML string
    $xml_payload = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<NCDC_Security_Tracker>\n";
    foreach ($tracker_data as $key => $value) {
        $xml_payload .= "  <$key>$value</$key>\n";
    }
    $xml_payload .= "</NCDC_Security_Tracker>";

    // Bury this file deep inside the document's structure where no normal user will ever look
    $zip->addFromString('word/media/ncdc_sys_data.xml', $xml_payload); 
    // Note: Use 'ppt/media/ncdc_sys_data.xml' if it is a PowerPoint file.
    
    $zip->close();
} else {
    die("Error: Failed to secure document for transport.");
}

// 6. Push the modified file to the user's browser
$download_filename = "NCDC_Secure_" . str_replace(' ', '_', $share_data['title']) . "." . pathinfo($original_file, PATHINFO_EXTENSION);

header('Content-Description: File Transfer');
header('Content-Type: application/octet-stream');
header('Content-Disposition: attachment; filename="' . $download_filename . '"');
header('Expires: 0');
header('Cache-Control: must-revalidate');
header('Pragma: public');
header('Content-Length: ' . filesize($temp_file));

// Clear output buffer and read the file
ob_clean();
flush();
readfile($temp_file);

// 7. Destroy the temporary file immediately after sending
unlink($temp_file);
exit;
?>